/-
Copyright (c) 2018 Chris Hughes. All rights reserved.
Released under Apache 2.0 license as described in the file LICENSE.
Authors: Chris Hughes, Yaël Dillies
-/
import Mathlib.Algebra.GeomSum
import Mathlib.Algebra.Order.Archimedean
import Mathlib.Algebra.Order.CauSeq.Basic

#align_import data.real.cau_seq from "leanprover-community/mathlib"@"9116dd6709f303dcf781632e15fdef382b0fc579"

/-!
# Cauchy sequences and big operators

This file proves some more lemmas about basic Cauchy sequences that involve finite sums.
-/

open Finset IsAbsoluteValue
open scoped BigOperators

namespace IsCauSeq
variable {α β : Type*} [LinearOrderedField α] [Ring β] {abv : β → α} [IsAbsoluteValue abv]
  {f g : ℕ → β} {a : ℕ → α}

lemma of_abv_le (n : ℕ) (hm : ∀ m, n ≤ m → abv (f m) ≤ a m) :
    IsCauSeq abs (fun n ↦ ∑ i in range n, a i) → IsCauSeq abv fun n ↦ ∑ i in range n, f i := by
  intro hg ε ε0
  cases' hg (ε / 2) (div_pos ε0 (by norm_num)) with i hi
  exists max n i
  intro j ji
  have hi₁ := hi j (le_trans (le_max_right n i) ji)
  have hi₂ := hi (max n i) (le_max_right n i)
  have sub_le :=
    abs_sub_le (∑ k in range j, a k) (∑ k in range i, a k) (∑ k in range (max n i), a k)
  have := add_lt_add hi₁ hi₂
  rw [abs_sub_comm (∑ k in range (max n i), a k), add_halves ε] at this
  refine' lt_of_le_of_lt (le_trans (le_trans _ (le_abs_self _)) sub_le) this
  generalize hk : j - max n i = k
  clear this hi₂ hi₁ hi ε0 ε hg sub_le
  rw [tsub_eq_iff_eq_add_of_le ji] at hk
  rw [hk]
  dsimp only
  clear hk ji j
  induction' k with k' hi
  · simp [abv_zero abv]
  simp only [Nat.succ_add, Nat.succ_eq_add_one, Finset.sum_range_succ_comm]
  simp only [add_assoc, sub_eq_add_neg]
  refine le_trans (abv_add _ _ _) ?_
  simp only [sub_eq_add_neg] at hi
  exact add_le_add (hm _ (le_add_of_nonneg_of_le (Nat.zero_le _) (le_max_left _ _))) hi
#align is_cau_series_of_abv_le_cau IsCauSeq.of_abv_le

lemma of_abv (hf : IsCauSeq abs fun m ↦ ∑ n in range m, abv (f n)) :
    IsCauSeq abv fun m ↦ ∑ n in range m, f n :=
  hf.of_abv_le 0 fun _ _ ↦ le_rfl
#align is_cau_series_of_abv_cau IsCauSeq.of_abv

theorem _root_.cauchy_product (ha : IsCauSeq abs fun m ↦ ∑ n in range m, abv (f n))
    (hb : IsCauSeq abv fun m ↦ ∑ n in range m, g n) (ε : α) (ε0 : 0 < ε) :
    ∃ i : ℕ, ∀ j ≥ i,
      abv ((∑ k in range j, f k) * ∑ k in range j, g k -
        ∑ n in range j, ∑ m in range (n + 1), f m * g (n - m)) < ε := by
  let ⟨P, hP⟩ := ha.bounded
  let ⟨Q, hQ⟩ := hb.bounded
  have hP0 : 0 < P := lt_of_le_of_lt (abs_nonneg _) (hP 0)
  have hPε0 : 0 < ε / (2 * P) := div_pos ε0 (mul_pos (show (2 : α) > 0 by norm_num) hP0)
  let ⟨N, hN⟩ := hb.cauchy₂ hPε0
  have hQε0 : 0 < ε / (4 * Q) :=
    div_pos ε0 (mul_pos (show (0 : α) < 4 by norm_num) (lt_of_le_of_lt (abv_nonneg _ _) (hQ 0)))
  let ⟨M, hM⟩ := ha.cauchy₂ hQε0
  refine ⟨2 * (max N M + 1), fun K hK ↦ ?_⟩
  have h₁ :
    (∑ m in range K, ∑ k in range (m + 1), f k * g (m - k)) =
      ∑ m in range K, ∑ n in range (K - m), f m * g n :=
    by simpa using sum_range_diag_flip K fun m n ↦ f m * g n
  have h₂ :
    (fun i ↦ ∑ k in range (K - i), f i * g k) = fun i ↦ f i * ∑ k in range (K - i), g k := by
    simp [Finset.mul_sum]
  have h₃ :
    ∑ i in range K, f i * ∑ k in range (K - i), g k =
      ∑ i in range K, f i * (∑ k in range (K - i), g k - ∑ k in range K, g k) +
        ∑ i in range K, f i * ∑ k in range K, g k :=
    by rw [← sum_add_distrib]; simp [(mul_add _ _ _).symm]
  have two_mul_two : (4 : α) = 2 * 2 := by norm_num
  have hQ0 : Q ≠ 0 := fun h ↦ by simp [h, lt_irrefl] at hQε0
  have h2Q0 : 2 * Q ≠ 0 := mul_ne_zero two_ne_zero hQ0
  have hε : ε / (2 * P) * P + ε / (4 * Q) * (2 * Q) = ε := by
    rw [← div_div, div_mul_cancel _ (Ne.symm (ne_of_lt hP0)), two_mul_two, mul_assoc, ← div_div,
      div_mul_cancel _ h2Q0, add_halves]
  have hNMK : max N M + 1 < K :=
    lt_of_lt_of_le (by rw [two_mul]; exact lt_add_of_pos_left _ (Nat.succ_pos _)) hK
  have hKN : N < K :=
    calc
      N ≤ max N M := le_max_left _ _
      _ < max N M + 1 := Nat.lt_succ_self _
      _ < K := hNMK
  have hsumlesum :
      (∑ i in range (max N M + 1),
        abv (f i) * abv ((∑ k in range (K - i), g k) - ∑ k in range K, g k)) ≤
      ∑ i in range (max N M + 1), abv (f i) * (ε / (2 * P)) := by
    gcongr with m hmJ
    refine le_of_lt $ hN (K - m) (le_tsub_of_add_le_left $ hK.trans' ?_) K hKN.le
    rw [two_mul]
    gcongr
    · exact (mem_range.1 hmJ).le
    · exact Nat.le_succ_of_le (le_max_left _ _)
  have hsumltP : (∑ n in range (max N M + 1), abv (f n)) < P :=
    calc
      (∑ n in range (max N M + 1), abv (f n)) = |∑ n in range (max N M + 1), abv (f n)| :=
        Eq.symm (abs_of_nonneg (sum_nonneg fun x _ ↦ abv_nonneg abv (f x)))
      _ < P := hP (max N M + 1)

  rw [h₁, h₂, h₃, sum_mul, ← sub_sub, sub_right_comm, sub_self, zero_sub, abv_neg abv]
  refine' lt_of_le_of_lt (IsAbsoluteValue.abv_sum _ _ _) _
  suffices
    (∑ i in range (max N M + 1),
          abv (f i) * abv ((∑ k in range (K - i), g k) - ∑ k in range K, g k)) +
        ((∑ i in range K, abv (f i) * abv ((∑ k in range (K - i), g k) - ∑ k in range K, g k)) -
          ∑ i in range (max N M + 1),
            abv (f i) * abv ((∑ k in range (K - i), g k) - ∑ k in range K, g k)) <
      ε / (2 * P) * P + ε / (4 * Q) * (2 * Q) by
    rw [hε] at this
    simpa [abv_mul abv] using this
  gcongr
  · exact lt_of_le_of_lt hsumlesum
        (by rw [← sum_mul, mul_comm]; gcongr)
  rw [sum_range_sub_sum_range (le_of_lt hNMK)]
  calc
    (∑ i in (range K).filter fun k ↦ max N M + 1 ≤ k,
          abv (f i) * abv ((∑ k in range (K - i), g k) - ∑ k in range K, g k)) ≤
        ∑ i in (range K).filter fun k ↦ max N M + 1 ≤ k, abv (f i) * (2 * Q) := by
        gcongr
        rw [sub_eq_add_neg]
        refine' le_trans (abv_add _ _ _) _
        rw [two_mul, abv_neg abv]
        gcongr <;> exact le_of_lt (hQ _)
    _ < ε / (4 * Q) * (2 * Q) := by
        rw [← sum_mul, ← sum_range_sub_sum_range (le_of_lt hNMK)]
        have := lt_of_le_of_lt (abv_nonneg _ _) (hQ 0)
        gcongr
        exact (le_abs_self _).trans_lt $ hM _ ((Nat.le_succ_of_le (le_max_right _ _)).trans hNMK.le)
          _  $ Nat.le_succ_of_le $ le_max_right _ _
#align cauchy_product cauchy_product

variable [Archimedean α]

lemma of_decreasing_bounded (f : ℕ → α) {a : α} {m : ℕ} (ham : ∀ n ≥ m, |f n| ≤ a)
    (hnm : ∀ n ≥ m, f n.succ ≤ f n) : IsCauSeq abs f := fun ε ε0 ↦ by
  classical
  let ⟨k, hk⟩ := Archimedean.arch a ε0
  have h : ∃ l, ∀ n ≥ m, a - l • ε < f n :=
    ⟨k + k + 1, fun n hnm ↦
      lt_of_lt_of_le (show a - (k + (k + 1)) • ε < -|f n| from
          lt_neg.1 <| (ham n hnm).trans_lt
              (by
                rw [neg_sub, lt_sub_iff_add_lt, add_nsmul, add_nsmul, one_nsmul]
                exact add_lt_add_of_le_of_lt hk (lt_of_le_of_lt hk (lt_add_of_pos_right _ ε0))))
        (neg_le.2 <| abs_neg (f n) ▸ le_abs_self _)⟩
  let l := Nat.find h
  have hl : ∀ n : ℕ, n ≥ m → f n > a - l • ε := Nat.find_spec h
  have hl0 : l ≠ 0 := fun hl0 ↦
    not_lt_of_ge (ham m le_rfl)
      (lt_of_lt_of_le (by have := hl m (le_refl m); simpa [hl0] using this) (le_abs_self (f m)))
  cases' not_forall.1 (Nat.find_min h (Nat.pred_lt hl0)) with i hi
  rw [not_imp, not_lt] at hi
  exists i
  intro j hj
  have hfij : f j ≤ f i := (Nat.rel_of_forall_rel_succ_of_le_of_le (· ≥ ·) hnm hi.1 hj).le
  rw [abs_of_nonpos (sub_nonpos.2 hfij), neg_sub, sub_lt_iff_lt_add']
  calc
    f i ≤ a - Nat.pred l • ε := hi.2
    _ = a - l • ε + ε := by
      conv =>
        rhs
        rw [← Nat.succ_pred_eq_of_pos (Nat.pos_of_ne_zero hl0), succ_nsmul', sub_add,
          add_sub_cancel]
    _ < f j + ε := add_lt_add_right (hl j (le_trans hi.1 hj)) _
#align is_cau_of_decreasing_bounded IsCauSeq.of_decreasing_bounded

lemma of_mono_bounded (f : ℕ → α) {a : α} {m : ℕ} (ham : ∀ n ≥ m, |f n| ≤ a)
    (hnm : ∀ n ≥ m, f n ≤ f n.succ) : IsCauSeq abs f :=
  (of_decreasing_bounded _ (by simpa using ham) $ by simpa using hnm).of_neg
#align is_cau_of_mono_bounded IsCauSeq.of_mono_bounded

lemma geo_series [Nontrivial β] (x : β) (hx1 : abv x < 1) :
    IsCauSeq abv fun n ↦ ∑ m in range n, x ^ m := by
  have hx1' : abv x ≠ 1 := fun h ↦ by simp [h, lt_irrefl] at hx1
  refine of_abv ?_
  simp only [abv_pow abv, geom_sum_eq hx1']
  conv in _ / _ => rw [← neg_div_neg_eq, neg_sub, neg_sub]
  have : 0 < 1 - abv x := sub_pos.2 hx1
  refine' @of_mono_bounded _ _ _ _ ((1 : α) / (1 - abv x)) 0 _ _
  · intro n _
    rw [abs_of_nonneg]
    gcongr
    · exact sub_le_self _ (abv_pow abv x n ▸ abv_nonneg _ _)
    refine' div_nonneg (sub_nonneg.2 _) (sub_nonneg.2 <| le_of_lt hx1)
    exact pow_le_one _ (by positivity) hx1.le
  · intro n _
    rw [← one_mul (abv x ^ n), pow_succ]
    gcongr
#align is_cau_geo_series IsCauSeq.geo_series

lemma geo_series_const (a : α) {x : α} (hx1 : |x| < 1) :
    IsCauSeq abs fun m ↦ ∑ n in range m, (a * x ^ n) := by
  simpa [mul_sum, Pi.mul_def] using (const a).mul (geo_series x hx1)
#align is_cau_geo_series_const IsCauSeq.geo_series_const

lemma series_ratio_test {f : ℕ → β} (n : ℕ) (r : α) (hr0 : 0 ≤ r) (hr1 : r < 1)
    (h : ∀ m, n ≤ m → abv (f m.succ) ≤ r * abv (f m)) :
    IsCauSeq abv fun m ↦ ∑ n in range m, f n := by
  have har1 : |r| < 1 := by rwa [abs_of_nonneg hr0]
  refine (geo_series_const (abv (f n.succ) * r⁻¹ ^ n.succ) har1).of_abv_le n.succ fun m hmn ↦ ?_
  obtain rfl | hr := hr0.eq_or_lt
  · have m_pos := lt_of_lt_of_le (Nat.succ_pos n) hmn
    have := h m.pred (Nat.le_of_succ_le_succ (by rwa [Nat.succ_pred_eq_of_pos m_pos]))
    simpa [Nat.succ_pred_eq_of_pos m_pos, pow_succ] using this
  generalize hk : m - n.succ = k
  replace hk : m = k + n.succ := (tsub_eq_iff_eq_add_of_le hmn).1 hk
  induction' k with k ih generalizing m n
  · rw [hk, Nat.zero_add, mul_right_comm, inv_pow _ _, ← div_eq_mul_inv, mul_div_cancel]
    positivity
  · have kn : k + n.succ ≥ n.succ := by
      rw [← zero_add n.succ]; exact add_le_add (Nat.zero_le _) (by simp)
    erw [hk, Nat.succ_add, pow_succ' r, ← mul_assoc]
    refine
      le_trans (by rw [mul_comm] <;> exact h _ (Nat.le_of_succ_le kn))
        (mul_le_mul_of_nonneg_right ?_ hr0)
    exact ih _ h _ (by simp) rfl
#align series_ratio_test IsCauSeq.series_ratio_test

end IsCauSeq
